#!/bin/bash
# Author: Steven Shiau <steven _at_ nchc org tw>
# License: GPL
#
# Note: This code is modified from http://cdprojekte.mattiasschlenker.de/Public/DSL-frominitrd/2.2b-0.0/script/pxedsl.sh
# Ref:
# http://news.mattiasschlenker.de/2006/02/22/pxe-booting-damnsmalllinux/#more-10
#
# SL: Small Linux (Damn Small Linux, Puppy Linux, INSERT or PLD)
#
. /opt/drbl/sbin/drbl-conf-functions

#
check_if_root

# Settings
SL_ISO_URL_EXAMPLE="http://free.nchc.org.tw/puppylinux/puppy-2.12-seamonkey.iso"

# List all supported SL_NAME
supported_SL_dists="DSL PuppyLinux INSERT PLD Debian-live GeeXbox PUD-Linux"

# The buffer ratio for iso to container
buffer_ratio_for_iso_to_container="1.25"

# The default estimation ratio for client's ram to use this small linux, 
# we estimate it as 3 times of the initrd. For some distribution we might overwrite this value if necessary.
client_ram_to_initrd_ratio="3"

# Functions
USAGE() {
  echo "Load small GNU/Linux, including Damn Small Linux, Puppy Linux, PLD Rescue CD or INSERT Linux to DRBL environment."
  echo "Usage: $0 [OPTION] [SL-ISO|SL-INDEX]"
  echo "OPTION:"
  language_help_prompt_by_idx_no
  echo "-i, --install ISO:        Load Small Linux ISO into DRBL environment, you must put the iso file in the current working dir." 
  echo "-d, --distribution DIST:  Assign the small GNU/Linux distribution as DIST. Available names are: $supported_SL_dists."
  echo "-u, --uninstall DIST:     Uninstall Small Linux DIST."
  echo "-V, --dist-version VER:   Assign the version of small GNU/Linux version number as VER."
  echo "-v, --verbose:     Verbose mode."
  echo "SL-ISO is one of $supported_SL_dists ISO file, used with installation."
  echo "S-L-INDEX is one of $supported_SL_dists, used with uninstallation."
  echo "Ex: To load DSL Linux, run '$0 -i dsl-3.3.iso'"
  echo "    To load PuppyLinux, run '$0 -i puppy-2.16-seamonkey-fulldrivers.iso'"
  echo "    To load INSERT Linux, run '$0 -i INSERT-1.3.9a_en.iso'"
  echo "    To load PLD Rescue CD, run '$0 -i rescue.iso'"
  echo "    To load Debian live CD, run '$0 -i debian-live-etch-i386-minimal.iso'"
  echo "    To load GeeXbox CD, run '$0 -i geexbox-1.0-en.i386.iso'"
  echo "    To load PUD GNU/Linux CD, run '$0 -i PUD-0.4.6.10.iso'"
  for isl in $supported_SL_dists; do
    echo "    To remove $isl, run '$0 -u $isl'"
  done
  echo "    To remove all Small Linux, run '$0 -u all'"
}
# get rootfile
get_SL_rootfile(){
  # this function must be used after iso is mounted, we need to find version inside it for some distribution (such as PLD).
  case "$SL_OS" in
    DSL|dsl|DamnSmallLinux)
       rootfile="$rootfile_path/KNOPPIX"
       # DSL does not have any extra file, the only one is KNOPPIX
       extra_sys_file=""
       ;;
    Puppy|puppy|PuppyLinux)
       rootfile="$rootfile_path/pup_${SL_VER}.sfs"
       # Puppy 2.12, there is a /zdrv_212.sfs
       [ -f "$isomnt/$rootfile_path/zdrv_${SL_VER}.sfs" ] && extra_sys_file="$rootfile_path/zdrv_${SL_VER}.sfs"
       ;;
    Insert|insert|INSERT)
       rootfile="$rootfile_path/INSERT"
       # INSERT does not have any extra file, the only one is INSERT
       extra_sys_file=""
       ;;
    PLD|pld)
       # PLD is diskless ready, so no rootfile in iso.
       rootfile=""
       extra_sys_file=""
       ;;
    debian-live-etch|Debian-live)
       rootfile="$rootfile_path/filesystem.squashfs"
       # debian-live-etch does not have any extra file, the only one is filesystem.squashfs 
       extra_sys_file=""
       ;;
    GeeXbox|geexbox)
       # The * is important, we just want files, directory GEEXBOX/ won't be appended in target dir.
       rootfile="$rootfile_path/GEEXBOX/*"
       ;;
    PUD|pud)
       rootfile="$rootfile_path/filesystem.squashfs"
       # PUD does not have any extra file, the only one is filesystem.squashfs 
       extra_sys_file=""
       ;;
  esac
} # end of prepare_param_in_pxe_cfg
#
prepare_param_in_pxe_cfg(){
  # This function must be used after ramdisk_size is calculated.
  case "$SL_OS" in
    DSL|dsl|DamnSmallLinux)
       append_param_in_pxe_cfg="ramdisk_size=$ramdisk_size init=/etc/init lang=us apm=power-off vga=791 initrd=$nbi_initrd nomce noapic quiet BOOT_IMAGE=knoppix frominitrd"
       ;;
    Puppy|puppy|PuppyLinux)
       case "$SL_VER" in 
         211|212|213|214) 
	    bootparam_trigger="frominitrd"
            use_modified_init="yes"
	    ;;
         *) # from version 216, init is ready when released, and no bootparam_trigger is necessary.
	    bootparam_trigger=""
            use_modified_init="no"
	    ;;
       esac
       append_param_in_pxe_cfg="ramdisk_size=$ramdisk_size root=/dev/ram0 initrd=$nbi_initrd loglevel=3 $bootparam_trigger"
       ;;
    Insert|insert|INSERT)
       append_param_in_pxe_cfg="ramdisk_size=$ramdisk_size init=/etc/init lang=en apm=power-off vga=773 initrd=$nbi_initrd nomce noapic dma BOOT_IMAGE=insert frominitrd"
       ;;
    PLD|pld)
       append_param_in_pxe_cfg="ramdisk_size=$ramdisk_size initrd=$nbi_initrd root=/dev/ram0"
       ;;
    debian-live-etch|Debian-live)
       append_param_in_pxe_cfg="initrd=$nbi_initrd boot=casper frominitrd"
       ;;
    GeeXbox|geexbox)
       append_param_in_pxe_cfg="ramdisk_size=$ramdisk_size initrd=$nbi_initrd root=/dev/ram0 rw init=linuxrc boot=frominitrd lang=en remote=atiusb receiver=atiusb keymap=qwerty splash=silent vga=773 video=vesafb:ywrap,mtrr"
       ;;
    PUD|pud)
       append_param_in_pxe_cfg="initrd=$nbi_initrd boot=casper show-cow wm=xfce4 frominitrd"
       ;;
  esac
} # end of prepare_param_in_pxe_cfg
#
create_linuxrc_pxe_initrd_with_rootfs_inserted(){
# prepare the size for initrd.
iso_size="$(stat -c "%s" $SL_ISO)" # unit: bytes
container_need_in_KB="$(echo "scale=0; $iso_size * $buffer_ratio_for_iso_to_container / 1024.0" | bc -l)" # unit: KB
client_RAM_required="$(echo "scale=0; $container_need_in_KB * $client_ram_to_initrd_ratio / 1024.0" | bc -l)"
# ramdisk_size is for client booting use (kernel parameter).
ramdisk_size="$container_need_in_KB"
echo "New NBI initrd size (uncompressed): $container_need_in_KB KB. Use ramdisk_size=$ramdisk_size in bootparam."
[ "$BOOTUP" = "color" ] && $SETCOLOR_WARNING
echo "$msg_RAM_size_for_SL_drbl_client: $client_RAM_required MB."
[ "$BOOTUP" = "color" ] && $SETCOLOR_NORMAL
echo "Creating the modified ramdisk will take a while, please be patient."

# Unpack initrd
echo 'Creating container...'
dd if=/dev/zero of=$wd/initrd.img bs=1024 count=$container_need_in_KB
echo 'Formatting container...'
mkfs.ext2 -q -F -L "${SL_NAME}-initrd" $wd/initrd.img
echo 'Unpacking old initrd...'
$uncompress_to_stdout "$oldinitrd" > $wd/oldinitrd.img
mkdir $wd/initrd.tmp
mkdir $wd/oldinitrd.tmp
# Mount initrd
mount -o loop $wd/initrd.img $wd/initrd.tmp
mount -o loop $wd/oldinitrd.img $wd/oldinitrd.tmp
#
if [ -z "$SL_VER" ]; then
  if [ -e $wd/oldinitrd.tmp/PUPPYVERSION ]; then
    SL_VER="$(cat $wd/oldinitrd.tmp/PUPPYVERSION)"
  elif [ -e $isomnt/GEEXBOX/etc/version ]; then
    # only if the modified version exists, otherwise we ust default one.
    SL_VER_TMP="$(cat $isomnt/GEEXBOX/etc/version)"
    if [ -e "$DRBL_SCRIPT_PATH/setup/files/$SL_NAME/$SL_VER_TMP/linuxrc.$SL_NAME-$SL_VER_TMP.drbl" ]; then
      SL_VER="$SL_VER_TMP"
    else
      SL_VER="default"
    fi
  else
    SL_VER="default"
  fi
fi

#
get_SL_rootfile
prepare_param_in_pxe_cfg

echo 'Copying content of the old initrd...'
( cp -a $wd/oldinitrd.tmp/* $wd/initrd.tmp )
mkdir -p $wd/initrd.tmp/$rootfile_path_in_initrd
echo 'Copying root image...'
cp -af $isomnt/$rootfile $wd/initrd.tmp/$rootfile_path_in_initrd/
if [ -n "$extra_sys_file" ]; then
  cp -af $isomnt/$extra_sys_file $wd/initrd.tmp/$rootfile_path_in_initrd/
fi
if [ "$use_modified_init" = "yes" ]; then
  echo "Copying modified linuxrc from $DRBL_SCRIPT_PATH/setup/files/$SL_NAME/$SL_VER/..."
  cp -f $DRBL_SCRIPT_PATH/setup/files/$SL_NAME/$SL_VER/linuxrc.$SL_NAME-$SL_VER.drbl $wd/initrd.tmp/linuxrc
else
  echo "The linuxrc from $SL_FULL_NAME is ready for PXE boot."
fi

# Umount initrd
umount $wd/initrd.tmp
umount $wd/oldinitrd.tmp

# Pack the initrd again:
echo 'Packing initrd...'
$compress_to_stdout $wd/initrd.img > $pxecfg_pd/$nbi_initrd

# Remove the initrd
echo 'Removing temporary copy of initrd...'
rm $wd/initrd.img
# Remove the old original initrd
echo 'Removing temporary copy of original initrd...'
rm $wd/oldinitrd.img

# Copy the linux kernel
cp -f "$kernel" $pxecfg_pd/$nbi_kernel

} # end of create_linuxrc_pxe_initrd_with_rootfs_inserted

#
put_pxe_initrd_without_rootfs_inserted(){
  cp -f $oldinitrd $pxecfg_pd/$nbi_initrd
  initrd_size="$(stat -c "%s" $pxecfg_pd/$nbi_initrd)" # unit: bytes
  initrd_size_in_KB=$(echo "scale=0; $initrd_size / 1024.0" | bc -l)
  client_RAM_required="$(echo "scale=0; $initrd_size_in_KB * $client_ram_to_initrd_ratio / 1024.0" | bc -l)"
  # ramdisk_size is for client booting use (kernel parameter).
  ramdisk_size="$initrd_size_in_KB"
  echo "New NBI initrd size (uncompressed): $initrd_size_in_KB KB. Use ramdisk_size=$ramdisk_size in bootparam."
  [ "$BOOTUP" = "color" ] && $SETCOLOR_WARNING
  echo "$msg_RAM_size_for_SL_drbl_client: $client_RAM_required MB."
  [ "$BOOTUP" = "color" ] && $SETCOLOR_NORMAL
  get_SL_rootfile
  prepare_param_in_pxe_cfg
  # Copy the linux kernel
  cp -f $kernel $pxecfg_pd/$nbi_kernel
  chmod 644 $pxecfg_pd/$nbi_kernel $pxecfg_pd/$nbi_initrd
} # end of put_pxe_initrd_without_rootfs_inserted
#
create_casper_pxe_initrd_with_rootfs_inserted(){
  case "$SL_OS" in
    debian-live-etch|Debian-live)
      [ -z "$SL_VER" ] && SL_VER="etch"
      get_SL_rootfile
      # it's initramfs, so no more loop mount, just use cpio
      mkdir $wd/initrd.tmp
      (cd $wd/initrd.tmp; $uncompress_to_stdout "$oldinitrd" | cpio -idm)
      mkdir -p $wd/initrd.tmp/$rootfile_path_in_initrd
      echo "Copying root image. $msg_this_might_take_several_minutes"
      cp -f $isomnt/$rootfile $wd/initrd.tmp/$rootfile_path_in_initrd/
      if [ "$use_modified_init" = "yes" ]; then
        echo "Copying modified casper from $DRBL_SCRIPT_PATH/setup/files/$SL_NAME/$SL_VER/..."
        cp -f $DRBL_SCRIPT_PATH/setup/files/$SL_NAME/$SL_VER/casper.$SL_NAME-$SL_VER.drbl $wd/initrd.tmp/scripts/casper
      fi
      # create initramfs
      echo "Creating initramfs $pxecfg_pd/$nbi_initrd, $msg_this_might_take_several_minutes"
      ( cd $wd/initrd.tmp
        find . | cpio --quiet -o -H newc | gzip -9 > $pxecfg_pd/$nbi_initrd
      )
      initramfs_size="$(stat -c "%s" $pxecfg_pd/$nbi_initrd)" # unit: bytes
      initramfs_size_in_KB=$(echo "scale=0; $initramfs_size / 1024.0" | bc -l)
      client_RAM_required="$(echo "scale=0; $initramfs_size_in_KB * $client_ram_to_initrd_ratio / 1024.0" | bc -l)"
      # initramfs, we do not have to assign ramdisk_size in bootparam.
      echo "The new initrd size is: $initramfs_size_in_KB KB."
      [ "$BOOTUP" = "color" ] && $SETCOLOR_WARNING
      echo "$msg_RAM_size_for_SL_drbl_client: $client_RAM_required MB."
      [ "$BOOTUP" = "color" ] && $SETCOLOR_NORMAL
      prepare_param_in_pxe_cfg
      cp -f $kernel $pxecfg_pd/$nbi_kernel
      chmod 644 $pxecfg_pd/$nbi_kernel $pxecfg_pd/$nbi_initrd
      ;;
    PUD|pud)
      if [ -z "$SL_VER" ]; then
        # iso file name is like PUD-0.4.6.10.iso, try to guess.
        SL_VER="$(echo $SL_ISO | sed -e "s/^PUD-//g" -e "s/.iso$//g")"
      fi
      get_SL_rootfile
      # it's initramfs, so no more loop mount, just use cpio
      mkdir $wd/initrd.tmp
      (cd $wd/initrd.tmp; $uncompress_to_stdout "$oldinitrd" | cpio -idm)
      mkdir -p $wd/initrd.tmp/$rootfile_path_in_initrd
      echo "Copying root image. $msg_this_might_take_several_minutes"
      cp -f $isomnt/$rootfile $wd/initrd.tmp/$rootfile_path_in_initrd/
      if [ "$use_modified_init" = "yes" ]; then
        echo "Copying modified casper from $DRBL_SCRIPT_PATH/setup/files/$SL_NAME/$SL_VER/..."
        cp -f $DRBL_SCRIPT_PATH/setup/files/$SL_NAME/$SL_VER/casper.$SL_NAME-$SL_VER.drbl $wd/initrd.tmp/scripts/casper
      else
        echo "The linuxrc from $SL_FULL_NAME is ready for PXE boot."
      fi
      # create initramfs
      echo "Creating initramfs $pxecfg_pd/$nbi_initrd, $msg_this_might_take_several_minutes"
      ( cd $wd/initrd.tmp
        find . | cpio --quiet -o -H newc | gzip -9 > $pxecfg_pd/$nbi_initrd
      )
      initramfs_size="$(stat -c "%s" $pxecfg_pd/$nbi_initrd)" # unit: bytes
      initramfs_size_in_KB=$(echo "scale=0; $initramfs_size / 1024.0" | bc -l)
      client_RAM_required="$(echo "scale=0; $initramfs_size_in_KB * $client_ram_to_initrd_ratio / 1024.0" | bc -l)"
      # initramfs, we do not have to assign ramdisk_size in bootparam.
      echo "The new initrd size is: $initramfs_size_in_KB KB."
      [ "$BOOTUP" = "color" ] && $SETCOLOR_WARNING
      echo "$msg_RAM_size_for_SL_drbl_client: $client_RAM_required MB."
      [ "$BOOTUP" = "color" ] && $SETCOLOR_NORMAL
      prepare_param_in_pxe_cfg
      cp -f $kernel $pxecfg_pd/$nbi_kernel
      chmod 644 $pxecfg_pd/$nbi_kernel $pxecfg_pd/$nbi_initrd
      ;;
  esac
} # end of create_casper_pxe_initrd_with_rootfs_inserted
#
umount_and_clean_tmp_working_dirs(){
  umount $isomnt
  [ -d "$isomnt" -a -n "$isomnt" ] && rm -rf "$isomnt"
  [ -d "$wd" -a -n "$wd" ] && rm -rf "$wd"
} # end of umount_and_clean_tmp_working_dirs
#
get_SL_OS_NAME(){
if [ "$(echo $SL_ISO | grep -i "dsl")" ]; then
  [ "$BOOTUP" = "color" ] && $SETCOLOR_WARNING
  echo "This ISO file is Damn Small Linux (DSL)."
  [ "$BOOTUP" = "color" ] && $SETCOLOR_NORMAL
  SL_OS="DSL"
elif [ "$(echo $SL_ISO | grep -i "puppy")" ]; then
  [ "$BOOTUP" = "color" ] && $SETCOLOR_WARNING
  echo "This ISO file is Puppy Linux."
  [ "$BOOTUP" = "color" ] && $SETCOLOR_NORMAL
  SL_OS="Puppy"
elif [ "$(echo $SL_ISO | grep -i "INSERT")" ]; then
  [ "$BOOTUP" = "color" ] && $SETCOLOR_WARNING
  echo "This ISO file is INSERT Linux."
  [ "$BOOTUP" = "color" ] && $SETCOLOR_NORMAL
  SL_OS="Insert"
elif [ "$(echo $SL_ISO | grep -i "rescue")" ]; then
  [ "$BOOTUP" = "color" ] && $SETCOLOR_WARNING
  echo "This ISO file is PLD Rescue CD."
  [ "$BOOTUP" = "color" ] && $SETCOLOR_NORMAL
  SL_OS="PLD"
elif [ "$(echo $SL_ISO | grep -i "debian-live-etch")" ]; then
  [ "$BOOTUP" = "color" ] && $SETCOLOR_WARNING
  echo "This ISO file is debian-live-etch CD."
  [ "$BOOTUP" = "color" ] && $SETCOLOR_NORMAL
  SL_OS="debian-live-etch"
elif [ "$(echo $SL_ISO | grep -i "GeeXbox")" ]; then
  [ "$BOOTUP" = "color" ] && $SETCOLOR_WARNING
  echo "This ISO file is GeeXbox CD."
  [ "$BOOTUP" = "color" ] && $SETCOLOR_NORMAL
  SL_OS="GeeXbox"
elif [ "$(echo $SL_ISO | grep -i "PUD")" ]; then
  [ "$BOOTUP" = "color" ] && $SETCOLOR_WARNING
  echo "This ISO file is PUD GNU/Linux CD."
  [ "$BOOTUP" = "color" ] && $SETCOLOR_NORMAL
  SL_OS="PUD"
else
  [ "$BOOTUP" = "color" ] && $SETCOLOR_FAILURE
  echo "Unknown Small Linux distribution! This script only works with Damn Small Linux or Puppy Linux! Program terminated!!!"
  [ "$BOOTUP" = "color" ] && $SETCOLOR_NORMAL
  exit 1
fi
} # end of get_SL_OS_NAME

#
install_SL(){
if [ -z "$SL_ISO" ]; then
  USAGE
  exit 1
fi

if [ ! -f "$SL_ISO" ]; then
  [ "$BOOTUP" = "color" ] && $SETCOLOR_FAILURE
  echo "$SL_ISO is NOT found!"
  echo "You have to prepare the iso file. For example, if you want to use PuppyLinux, you can get it via this command:"
  echo "wget $SL_ISO_URL_EXAMPLE"
  [ "$BOOTUP" = "color" ] && $SETCOLOR_NORMAL
  exit 1
fi

if [ -z "$(file $SL_ISO | grep -i "ISO 9660 CD-ROM filesystem data")" ]; then
  [ "$BOOTUP" = "color" ] && $SETCOLOR_FAILURE
  echo "$SL_ISO is not an ISO 9660 CD-ROM filesystem data file!"
  [ "$BOOTUP" = "color" ] && $SETCOLOR_NORMAL
  echo "Program terminated!"
  exit 1
fi

# get SL_OS name from iso file name
# Ugly, since it's not easy to know which one is which one, here we just judge them by file name.
[ -z "$SL_OS" ] && get_SL_OS_NAME

# Some settings for SL
# insert_rootfs=yes => we have to put the root/main filesystem from SL in initrd
# init_type: linuxrc or casper
# use_modified_init: yes or no.
# 2 types of init: linuxrc or casper. Some are not ready for PXE, we have to mofify, some are ready.
# Basically there are 6 types of SL:
# (1) linuxrc is ready for PXE, initrd is ready for PXE (insert_rootfs=no, use_modified_init=no). Actually if insert_rootfs=no, use_modified_init must be no. Ex: PLD
# (2) linuxrc is ready for PXE, initrd is NOT ready for PXE (insert_rootfs=yes, use_modified_init=no). Ex: Insert 1.39a or later
# (3) linuxrc is NOT ready for PXE, initrd is NOT ready for PXE (insert_rootfs=yesyes, use_modified_init=yes). Ex:DSL, PuppyLinux, GeeXbox
# (4) casper is ready for PXE, initrd is ready for PXE (insert_rootfs=no, use_modified_init=yes). Actually no such SL now (2007/02/19), maybe in the future when they accept our patch.
# (5) casper is NOT ready for PXE, initrd is NOT ready for PXE (insert_rootfs=yes, use_modified_init=yes). Ex: Debian Live, PUD.
# (6) casper is ready for PXE, initrd is NOT ready for PXE (insert_rootfs=yes, use_modified_init=yes). Actually no such SL now (2007/02/19), maybe in the future when they accept our patch.

case "$SL_OS" in
  DSL|dsl|DamnSmallLinux)
     # SL_NAME: in one word
     SL_NAME="DSL"
     SL_FULL_NAME="Damn Small Linux"
     # In DSL, boot kernel files are linux24 and minirt24.gz in /boot/isolinux/
     bootfiles_path="/boot/isolinux"
     # in DSL, it's cloop file /cdrom/KNOPPIX/KNOPPIX
     rootfile_path="/KNOPPIX"
     rootfile_path_in_initrd="/cdrom/KNOPPIX"
     insert_rootfs="yes"
     init_type="linuxrc"
     use_modified_init="yes"
     ;;
  Puppy|puppy|PuppyLinux)
     # SL_NAME: in one word
     SL_NAME="PuppyLinux"
     SL_FULL_NAME="Puppy Linux"
     # In PuppyLinux, boot kernel files are vmlinuz and initrd.gz in /
     bootfiles_path="/"
     # in PuppyLinux, it's squashfs, name is like pup_211.sfs in /
     rootfile_path="/"
     rootfile_path_in_initrd="/"
     insert_rootfs="yes"
     init_type="linuxrc"
     use_modified_init="yes"
     ;;
  Insert|insert|INSERT)
     # SL_NAME: in one word
     SL_NAME="INSERT"
     SL_FULL_NAME="Inside Security Rescue Toolkit (INSERT) Linux"
     # In INSERT, boot kernel files are vmlinuz and miniroot.lz in /isolinux/
     bootfiles_path="/isolinux"
     # In INSERT, it's squash file /cdrom/INSERT/INSERT
     rootfile_path="/INSERT"
     rootfile_path_in_initrd="/cdrom/INSERT"
     insert_rootfs="yes"
     init_type="linuxrc"
     # linuxrc in INSERT 1.38 or earlier is not ready for PXE (frominitrd), from 1.39a, they accepted our patch file for PXE.
     use_modified_init="no"
     # check if lzma exists
     if ! type lzma &>/dev/null; then
       [ "$BOOTUP" = "color" ] && $SETCOLOR_FAILURE
       echo "lzma is NOT found! You have to install it."
       [ "$BOOTUP" = "color" ] && $SETCOLOR_NORMAL
       echo "lzma is available in http://tukaani.org/lzma/download."
       echo "For Debian Etch or Ubuntu Edgy, just use 'apt-get install lzma' to install it."
       echo "For Mandriva 2007, just use 'urpmi lzma' to install it."
       echo "Program terminated!!!"
       exit 1
     fi
     ;;
  PLD|pld)
     # SL_NAME: in one word
     SL_NAME="PLD"
     SL_FULL_NAME="PLD Linux Rescue CD"
     # In PLD, boot kernel files are vmlinuz in /boot/isolinux/
     bootfiles_path="/boot/isolinux"
     # in PLDLinux, it's cpio, name is like rescue.cpi in /
     rootfile_path="/"
     rootfile_path_in_initrd="/"
     insert_rootfs="no"
     # init_type is not necessary here, but we still list it
     init_type="linuxrc"
     use_modified_init="no"
     ;;
  debian-live-etch|Debian-live)
     # SL_NAME: in one word
     SL_NAME="Debian-live"
     SL_FULL_NAME="Debian Live Etch"
     # In debian-live-etch, boot kernel files are vmlinuz in /isolinux/
     bootfiles_path="/isolinux"
     # in debian-live-etch, it's cpio, name is like /live_media/casper/filesystem.squashfs
     rootfile_path="/casper"
     rootfile_path_in_initrd="/live_media/casper"
     insert_rootfs="yes"
     init_type="casper"
     use_modified_init="yes"
     ;;
  GeeXbox|geexbox)
     # SL_NAME: in one word
     SL_NAME="GeeXbox"
     SL_FULL_NAME="GeeXbox"
     # In GeeXbox, boot kernel files are vmlinuz in /GEEXBOX/boot/
     bootfiles_path="/GEEXBOX/boot/"
     # in GeeXbox, no cloop/cpio file, just all in /GEEXBOX
     rootfile_path="/"
     rootfile_path_in_initrd="/"
     # ramdisk_size is for client booting use (kernel parameter).
     insert_rootfs="yes"
     init_type="linuxrc"
     use_modified_init="yes"
     ;;
  PUD|pud)
     # SL_NAME: in one word
     SL_NAME="PUD-Linux"
     SL_FULL_NAME="PUD Linux"
     # In PUD Linux, boot kernel files are vmlinuz in /
     bootfiles_path="/"
     # in PUD Linux, it's cpio, name is like /cdrom/casper/filesystem.squashfs
     rootfile_path="/casper"
     rootfile_path_in_initrd="/cdrom/casper"
     insert_rootfs="yes"
     init_type="casper"
     use_modified_init="yes"
     ;;
esac

#
echo "$msg_delimiter_star_line"
echo "$msg_this_script_will_create_SL_diskless: $SL_FULL_NAME"
echo -n "$msg_are_u_sure_u_want_to_continue [Y/n] "
read confirm_ans
case "$confirm_ans" in
  n|N|[nN][oO])
     echo "$msg_program_stop"
     exit 1
     ;;
  *)
     echo "$msg_ok_let_do_it"
     ;;
esac

#
isomnt="$(mktemp -d /tmp/sl.XXXXXX)"
wd="$(mktemp -d drbl_sl_wd.XXXXXX)"

mount -o loop $SL_ISO $isomnt

# Find kernel
echo "Finding the kernel and initrd from iso..."
for i in $isomnt/$bootfiles_path/*; do
  # sometimes it's "Linux kernel x86", sometimes it's "Linux x86 kernel"
  # In Ubuntu Sarge (file 4.17-2ubuntu1):
  # For DSL: linux24: Linux x86 kernel root=0x301-ro vga=normal, bzImage, version 2.4.26 (root@Knoppix) #1 SMP Sa
  # For Puppy: vmlinuz: Linux kernel x86 boot executable RO-rootFS, root_dev 0x341, swap_dev 0x1, Normal VGA
  # For INSERT: vmlinuz: Linux kernel x86 boot executable RO-rootFS, root_dev 0x806, swap_dev 0x1, Normal VGA
  # The problem is that, for some distribution, like INSERT, there is another file "memtest86": memtest86: Linux x86 kernel
  # For Clonezilla live iso, there is another one: eb.zli (etherboot)
  # For old file, like Debian Sarge (file 4.12-1) or RH9 (file-3.39-9):
  # In Debian Sarge: vmlinuz-2.6.8-2-686: x86 boot sector
  # In RH9: vmlinuz-2.4.20-28.9smp: x86 boot sector
  if [ -n "$(file $i | grep -iE "(Linux x86 kernel|Linux kernel x86 boot executable|x86 boot sector)")" ]; then
    # Ugly
    # we skip memtest
    if [ -z "$(echo $i | grep -iE "(memtest|eb.zli)")" ]; then
      kernel="$i"
      break
    fi
  fi
done
if [ -n "$kernel" ]; then
  echo "Found the kernel: $kernel"
else
  [ "$BOOTUP" = "color" ] && $SETCOLOR_FAILURE
  echo "The kernel does NOT exist! Program terminated!"
  [ "$BOOTUP" = "color" ] && $SETCOLOR_NORMAL
  umount_and_clean_tmp_working_dirs
  exit 1
fi

# Find initrd
# UGLY! Here we assume only one gzip/cpio file exists, and it's initrd.
for i in $isomnt/$bootfiles_path/* $isomnt/$rootfile_path/*; do
  if [ -n "$(file $i | grep -i "gzip compressed data")" ]; then
    oldinitrd="$i"
    uncompress_to_stdout="gunzip -c"
    compress_to_stdout="gzip -c"
    break
  elif [ -n "$(echo $i | grep -iE "\/miniroot.lz$")" ]; then
  # For INSERT, the initrd is miniroot.lz, which is lzma format. Most of the program file does not have the magic for that.
    oldinitrd="$i"

    # There are two types of lzma, although they all said they are 4.32/4.33
    # (1) MDV2007, Ubuntu Etch
    #   LZMA 4.32 Copyright (c) 1999-2005 Igor Pavlov  2005-12-09
    #   Usage:  LZMA <e|d> inputFile outputFile [<switches>...]
    #   e: encode file
    #	  d: decode file
    #   -si:    read data from stdin
    #   -so:    write data to stdout
    # (2) Debian Etch
    #   lzma 4.32.0beta3 Copyright (C) 2006 Ville Koskinen
    #   Based on LZMA SDK 4.43 Copyright (C) 1999-2006 Igor Pavlov
    #   -c --stdout       output to standard output
    #   -d --decompress   force decompression
    #   -z --compress     force compression
    if [ -n "$(lzma --h 2>&1 | grep -iEw "\-so:")" ]; then
      uncompress_to_stdout="lzma d -so"
      compress_to_stdout="lzma e -so"
    else
      uncompress_to_stdout="lzma -d -c -S .lz"
      compress_to_stdout="lzma -z -c"
    fi
    break
  elif [ -n "$(file $i | grep -i "ASCII cpio archive")" ]; then
    oldinitrd="$i"
    break
  fi
done
if [ -n "$oldinitrd" ]; then
  echo "Found the initrd: $oldinitrd"
else
  [ "$BOOTUP" = "color" ] && $SETCOLOR_FAILURE
  echo "The initrd does NOT exist! Program terminated!"
  [ "$BOOTUP" = "color" ] && $SETCOLOR_NORMAL
  umount_and_clean_tmp_working_dirs
  exit 1
fi
# we rename the NBI kernel and img, like dsl-linux24 and dsl-minirt24.gz
nbi_kernel="${SL_NAME}-${kernel##*/}"
nbi_initrd="${SL_NAME}-${oldinitrd##*/}"

#
case "$insert_rootfs" in
  yes)
      case "$init_type" in
        "linuxrc")
           create_linuxrc_pxe_initrd_with_rootfs_inserted
           ;;
        "casper")
           create_casper_pxe_initrd_with_rootfs_inserted
	   ;;
      esac
      ;;
  no)
      put_pxe_initrd_without_rootfs_inserted
      ;;
esac
#
umount_and_clean_tmp_working_dirs

# append the config file in pxelinux dir.
if [ -z "$(grep -E "^[[:space:]]*label[[:space:]]+$SL_NAME[[:space:]]*$" $PXELINUX_DIR/default)" ]; then
  echo "Append the $SL_FULL_NAME config in $PXELINUX_DIR/default..."
  cat <<-SL_PXE >> $PXELINUX_DIR/default 
label $SL_NAME
  # MENU DEFAULT
  # MENU HIDE
  MENU LABEL $SL_FULL_NAME (Ramdisk)
  # MENU PASSWD
  KERNEL $nbi_kernel
  APPEND $append_param_in_pxe_cfg

SL_PXE
fi
echo "$msg_delimiter_star_line"
[ "$BOOTUP" = "color" ] && $SETCOLOR_WARNING
echo "$msg_all_set_you_can_turn_on_clients"
echo "$msg_note! $msg_etherboot_5_4_is_required"
[ "$BOOTUP" = "color" ] && $SETCOLOR_NORMAL
} # end of install_SL

# 
uninstall_SL(){
  # convert
  case "$SL_TO_BE_REMOVED" in
    [aA][lL][lL])
      SL_TO_BE_REMOVED="$supported_SL_dists"
      ;;
  esac
  echo "Uninstalling installed Small Linux..."
  for id in $SL_TO_BE_REMOVED; do
    if [ -z "$(echo "$supported_SL_dists" | grep -Ew "$id")" ]; then
      echo "$SL_TO_BE_REMOVED is an unknown Small Linux! Program terminated!"
      exit 1
    fi
    # remove the block in pxelinux
    if [ -n "$(grep -E "^[[:space:]]*label[[:space:]]+$id[[:space:]]*$" $PXELINUX_DIR/default)" ]; then
      echo "Removing $id setting in $PXELINUX_DIR/default..."
      delete_label_block_pxe_img $id $PXELINUX_DIR/default
    fi
    # remove vmlinuz and initrd
    if [ -n "$(ls $pxecfg_pd/${id}-* 2>/dev/null)" ]; then
      echo "Removing installed $id if it exists..."
      rm -fv $pxecfg_pd/${id}-*
    fi
  done
} # end of uninstall_SL

#############
###  MAIN ###
#############
#
check_if_root

# Parse command-line options
while [ $# -gt 0 ]; do
  case "$1" in
    -l|--language)
            shift
            if [ -z "$(echo $1 |grep ^-.)" ]; then
              # skip the -xx option, in case 
              specified_lang="$1"
	      [ -z "$specified_lang" ] && USAGE && exit 1
              shift
            fi
            ;;
    -i|--install)
        shift; mode="install"
        if [ -z "$(echo $1 |grep ^-.)" ]; then
          # skip the -xx option, in case 
          SL_ISO="$1"
          [ -z "$SL_ISO" ] && USAGE && exit 1
	  shift
        fi
	;;
    -d|--distribution)
        shift;
        if [ -z "$(echo $1 |grep ^-.)" ]; then
          # skip the -xx option, in case 
          SL_OS="$1"
          [ -z "$SL_OS" ] && USAGE && exit 1
	  shift
        fi
	;;
    -u|--uninstall)
        shift; mode="uninstall"
        if [ -z "$(echo $1 |grep ^-.)" ]; then
          # skip the -xx option, in case 
          SL_TO_BE_REMOVED="$1"
          [ -z "$SL_TO_BE_REMOVED" ] && USAGE && exit 1
	  shift
        fi
        ;;
    -V|--dist-version)
        shift;
        if [ -z "$(echo $1 |grep ^-.)" ]; then
          # skip the -xx option, in case 
          SL_VER="$1"
          [ -z "$SL_VER" ] && USAGE && exit 1
	  shift
        fi
	;;
    -v|--verbose)
	verbose="on"
	verbose_opt="-v"
        shift ;;
    *)  USAGE && exit 1 ;;
  esac
done
# mode is essential
[ -z "$mode" ] && USAGE && exit 1

# Load the language file
ask_and_load_lang_set $specified_lang

# run it
case "$mode" in
   install) install_SL;;
   uninstall) uninstall_SL;;
esac
